#region References

using System;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using gov.va.med.vbecs.Common.Log;
using ServerTimers = System.Timers;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

#endregion

namespace gov.va.med.vbecs.DAL.HL7AL
{

	#region class HL7TestConnection

	/// <summary>
	/// CR 2940
	/// Russell Stephenson 3/21/2016 Deleted IsEnforcedException for defect# 286500 Fortify Findings 
	/// </summary>
	public class HL7TestConnection : IDisposable
	{
		private enum ConnectionState
		{
			Disconnected,
			Connecting,
			Connected,
			Disconnecting,
			Uninitialized,
			Reading,
			Error
		}

		private BlockingTimer _ackTimeoutTimer;
		private TcpClient _client;
		private NetworkStream _stream;
		private ConnectionState _connectionState = ConnectionState.Uninitialized;

		private IPAddress _ipAddress;
		private string _hostName;
		private int _portNumber;

		/// <summary>
		/// Constructor for HL7ClientConnection
		/// </summary>
		public HL7TestConnection(string hostName, int portNumber)
		{
			ResetConnectionStateVars();
			//
			_hostName = hostName;
			//
			_portNumber = portNumber;
		}

		/// <summary>
		/// Constructor for HL7ClientConnection
		/// </summary>
		public HL7TestConnection(IPAddress ipAddress, int portNumber)
		{
			ResetConnectionStateVars();
			//
			_ipAddress = ipAddress;
			//
			_portNumber = portNumber;
		}

		/// <summary>
		/// Connects to a HL7 Listener
		/// </summary>
		public void Connect()
		{
			lock( this )
			{
				if( this.IsConnected )
				{
					throw( new InvalidOperationException( "InvalidOperationException" ) );
				}
				//
				try
				{
					SetConnectionState( ConnectionState.Connecting );

					_client = new TcpClient();
					_client.SendTimeout = 10;
					_client.ReceiveTimeout = 10;

					if ( _hostName != null )
					{
						_client.Connect( _hostName, _portNumber );	
					}
					else
					{
						_client.Connect( _ipAddress, _portNumber );	
					}

					_stream = _client.GetStream();
				}
				catch( SocketException xcp )
				{
					Disconnect();
						
					throw( new Exception( "GeneralNetworkError", xcp ) );
				}
				catch( IOException xcp )
				{
					Disconnect();

					// This is required due to 'problematic' implementation of the
					// FCL's NetworkStream which catches every exception and wraps it as IOException
					if( !(xcp.InnerException is SocketException) )
						throw( xcp.InnerException );

					throw( new Exception( "GeneralNetworkError", xcp ) );
				}
				catch
				{
					Disconnect();
					throw;
				}

				SetConnectionState( ConnectionState.Connected );
			}
		}

		/// <summary>
		/// Disconnects from server with or without prior 
		/// notification (depending on parameter specified).
		/// </summary>
		private void Disconnect()
		{
			lock( this )
			{
				try
				{
					try
					{
						bool _wasConnected = IsConnected;

						SetConnectionState( ConnectionState.Disconnecting );

						if( _ackTimeoutTimer != null )
							_ackTimeoutTimer.Close();
						
					}
					finally
					{
						if( _client != null )
							_client.Close();
						
						if( _stream != null )
							_stream.Close();
					}
				}
				finally
				{
					ResetConnectionStateVars();
				}
			}
		}

		/// <summary>
		/// Indicates whether connection with HL7 Listener is established
		/// </summary>
		public bool IsConnected
		{
			get
			{
				lock( this )
					return _connectionState == ConnectionState.Connected;
			}
		}

		/// <summary>
		/// Finalization method closing the connection (disconnecting from server). 
		/// </summary>
		~HL7TestConnection()
		{
			Dispose( false );
		}

		/// <summary>
		/// Cleanup method needed to implement IDisposable
		/// </summary>
		public void Dispose()
		{
			GC.SuppressFinalize( this );
			Dispose( true );
		}

		/// <summary>
		/// Closes the connection and disposes of the object.
		/// </summary>
		public void Close()
		{
			Disconnect();
			Dispose();
		}

		/// <summary>
		/// Standard implementation of the dispose method. 
		/// </summary>
		/// <param name="disposing">
		///		Flag indicating if the disposition was invoked explicitly under normal 
		///		conditions (true) or forced upon object disposal (false).
		///	</param>
		private void Dispose( bool disposing )
		{
			lock( this )
			{
				if( disposing )
					Disconnect();

				ResetConnectionStateVars();
			}
		}

		/// <summary>
		/// Resets connection state and network related member variables to default disconnected state.
		/// </summary>
		private void ResetConnectionStateVars()
		{
			_client = null;
			_stream = null;
			_ackTimeoutTimer = null;		
			_ipAddress = null;
			_hostName = null;
			_portNumber = 0;
			//
			SetConnectionState( ConnectionState.Disconnected );
		}

		/// <summary>
		/// Sets connection state and notifies registered handlers if state change has occured.
		/// </summary>
		/// <param name="connectionStateToSet">New connection state to set.</param>
		private void SetConnectionState( ConnectionState connectionStateToSet )
		{
			lock( this )
			{
				ConnectionState _previousConnectionState = _connectionState;

				_connectionState = connectionStateToSet;

				bool _isNewStateConnected = IsConnectedState( connectionStateToSet );
			}
		}

		/// <summary>
		/// Determines whether a given connection state is available ('connected') state - 
		/// state in which connection may be used by outside callers.
		/// </summary>
		/// <param name="state"></param>
		/// <returns></returns>
		private bool IsConnectedState( ConnectionState state )
		{
			return state == ConnectionState.Connected;
		}
	}

	#endregion

	/// <summary>
	/// Summary description for Class1.
	/// </summary>
	public class HL7ClientConnection : IDisposable
	{
		private enum ConnectionState
		{
			Disconnected,
			Connecting,
			Connected,
			Disconnecting,
			Uninitialized,
			Reading,
			Error
		}

		private BlockingTimer _ackTimeoutTimer;
		private TcpClient _client;
		private NetworkStream _stream;
		private HL7Interface _interfaceConnectInfo;

		private readonly HL7ProtocolMessageFactory _messageFactory;
		private bool _requestSent;

        // Events Logger
        private readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

		private ConnectionState _connectionState = ConnectionState.Uninitialized;

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8285"> 
		///		<ExpectedInput>HL7Interface</ExpectedInput>
		///		<ExpectedOutput>HL7ClientConnection object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8286"> 
		///		<ExpectedInput>Null</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor for HL7ClientConnection
		/// </summary>
		/// <param name="interfaceConnectionInfo">HL7 Interface configuration information</param>
		public HL7ClientConnection( HL7Interface interfaceConnectionInfo  )
		{
			if( interfaceConnectionInfo == null )
				throw( new ArgumentNullException( "interfaceConnectionInfo" ) );

			_interfaceConnectInfo = interfaceConnectionInfo;

			_messageFactory = new HL7ProtocolMessageFactory();

			ResetConnectionStateVars();
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8287"> 
		///		<ExpectedInput>Untestable</ExpectedInput>
		///		<ExpectedOutput>Untestable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8288"> 
		///		<ExpectedInput>Untestable</ExpectedInput>
		///		<ExpectedOutput>Untestable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Connects to VistA HL7 Listener.
		/// </summary>
		public void Connect()
		{
			lock( this )
			{
				if( this.IsConnected )
				{
					throw( new InvalidOperationException( "InvalidOperationExeption" ) );
				}
				//
				try
				{
					SetConnectionState( ConnectionState.Connecting );

					_client = new TcpClient();
					_client.SendTimeout = _interfaceConnectInfo.AckTimeout;
					_client.ReceiveTimeout = _interfaceConnectInfo.AckTimeout;

					if( _interfaceConnectInfo.InterfaceDomainName != null )
					{
						_client.Connect( _interfaceConnectInfo.InterfaceDomainName, _interfaceConnectInfo.InterfacePortNumber );
					}
					else
					{
						_client.Connect( _interfaceConnectInfo.InterfaceIPAddress, _interfaceConnectInfo.InterfacePortNumber );	
					}

					_stream = _client.GetStream();
				}
				catch( SocketException xcp )
				{
					Disconnect();
					
					throw( new HL7NetworkErrorException( "GeneralNetworkError", xcp ) );
				}
				catch( IOException xcp )
				{
					Disconnect();

					// This is required due to 'problematic' implementation of the
					// FCL's NetworkStream which catches every exception and wraps it as IOException
					if( !(xcp.InnerException is SocketException) )
						throw( xcp.InnerException );

					throw( new HL7NetworkErrorException( "GeneralNetworkError", xcp ) );
				}
				catch
				{
					Disconnect();
					throw;
				}

				_requestSent = false;
				SetConnectionState( ConnectionState.Connected );
			}
		}

		/// <summary>
		/// Disconnects from VistA server with or without prior 
		/// notification (depending on parameter specified).
		/// </summary>
		private void Disconnect()
		{
			lock( this )
			{
				try
				{
					try
					{
						bool _wasConnected = IsConnected;

						SetConnectionState( ConnectionState.Disconnecting );

						if( _ackTimeoutTimer != null )
							_ackTimeoutTimer.Close();
					}
					finally
					{
						if( _client != null )
							_client.Close();
					
						if( _stream != null )
							_stream.Close();
					}
				}
				finally
				{
					ResetConnectionStateVars();
				}
			}
		}

		/// <summary>
		/// Sends specified VistALink message to the server 
		/// converting message to byte array and writing to the network stream 
		/// without starting/stopping heartbeat.
		/// </summary>
		/// <param name="messageToSend">Client message to send to remote server.</param>
		private void UntimedSendMessage( HL7ProtocolMessage messageToSend )
		{
			try
			{
				if( messageToSend == null )
					throw( new ArgumentNullException( "messageToSend" ) );

				if( _requestSent )
					throw( new InvalidOperationException( "PrematureWriteOccured" ) );


				byte[] _dataChunk = messageToSend.ToByteArray();
				_stream.Write( _dataChunk, 0, _dataChunk.Length );

				_requestSent = true;
			}
			catch( IOException xcp )
			{
				Disconnect();

				// This is required due to 'problematic' implementation of the
				// FCL's NetworkStream.Read / Write which catches every exception and wraps it as IOException
				if( !(xcp.InnerException is SocketException) )
					throw( xcp.InnerException );

				throw( new HL7NetworkErrorException( "GeneralNetworkError", xcp ) );
			}
			catch
			{
				Disconnect();
				throw;
			}
		}

		/// <summary>
		/// Receives message reading data from the network stream, parsing to XML and creating 
		/// message without starting/stopping heartbeat.
		/// CR 2961
		/// </summary>
		/// <returns>Received VistALink message of appropriate type.</returns>
		private HL7ProtocolMessage UntimedReceiveMessage()
		{
		    StringBuilder _msgString;

			try
			{
				if( !_requestSent )
					throw( new InvalidOperationException( "PrematureReadOccured" ) );

				byte[] _receiveBuffer = new byte[1];
				new MemoryStream( _receiveBuffer.Length );
				_msgString = new StringBuilder();
				bool _startBlockReceived = false;
				int _bytesRead = 0;
				for(;;)
				{
					_bytesRead = _stream.Read( _receiveBuffer, 0, _receiveBuffer.Length );
					
					if( _bytesRead <= 0 ) 					
						throw( new HL7Exception( "UnexpectedEOTOccured" ) );

					// Received start block but not end block
					if( _receiveBuffer[0] == HL7Interface.SOTByte )
					{
						_startBlockReceived = true;
					}

					// Received end block this time.  Write the rest to the stream and break out.
					if( _receiveBuffer[0] == HL7Interface.FSByte )
					{
						break;
					}

						// Received something in between start and end block.  Highly unlikely!
					else if( _receiveBuffer[0] != HL7Interface.SOTByte && _startBlockReceived )
					{
						_msgString.Append( Encoding.ASCII.GetChars( _receiveBuffer ) );
					}
				}
				_requestSent = false;
			}
			catch( IOException xcp )
			{
				Disconnect();

				// This is required due to 'problematic' implementation of the
				// FCL's NetworkStream.Read / Write which catches every exception and wraps it as IOException
				if( !(xcp.InnerException is SocketException) )
					throw( xcp.InnerException );

				throw( new HL7NetworkErrorException( "GeneralNetworkError", xcp ) );
			}
			catch
			{
				Disconnect();
				throw;
			}
	
			try
			{
				var msg = _messageFactory.ParseCreateHL7ProtocolMessage( _msgString.ToString() );

				// CR 2961
			    if (msg != null) return msg;
                throw (new HL7Exception(string.Concat("Unknown Response Message Type Received:\n\n", _msgString.ToString())));
			}
			catch
			{
				Disconnect();
				throw;
			}
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8289"> 
		///		<ExpectedInput>Untestable</ExpectedInput>
		///		<ExpectedOutput>Untestable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8290"> 
		///		<ExpectedInput>Untestable</ExpectedInput>
		///		<ExpectedOutput>Untestable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sends HL7 message to interface listener and receives response message. 
		/// </summary>
		/// <param name="clientMessage">Message to send.</param>
		/// <returns>Server response message.</returns>
		public HL7ProtocolMessage SendReceiveMessage( HL7ProtocolMessage clientMessage )
		{	
			lock( this )
			{
				if( !IsConnected )
					throw( new InvalidOperationException( "ConnectionIsNotYetOpen" ) );

				try
				{
					UntimedSendMessage( clientMessage );

					HL7ProtocolMessage result = UntimedReceiveMessage();
						
					Disconnect();

					return result;
				}
				catch
				{
					Disconnect();
					throw;
				}
			}
		}

		/// <summary>
		/// Finalization method closing the connection (disconnecting from server). 
		/// </summary>
		~HL7ClientConnection()
		{
			Dispose( false );
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8291"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8292"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Cleanup method needed to implement IDisposable
		/// </summary>
		public void Dispose()
		{
			GC.SuppressFinalize( this );
			Dispose( true );
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8293"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8294"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Closes the connection and disposes of the object.
		/// </summary>
		public void Close()
		{
			Disconnect();
			Dispose();
		}

		/// <summary>
		/// Standard implementation of the dispose method. 
		/// </summary>
		/// <param name="disposing">
		///		Flag indicating if the disposition was invoked explicitly under normal 
		///		conditions (true) or forced upon object disposal (false).
		///	</param>
		private void Dispose( bool disposing )
		{
			lock( this )
			{
				if( disposing )
					Disconnect();

				ResetConnectionStateVars();

				_interfaceConnectInfo = null;
			}
		}

		/// <summary>
		/// Resets connection state and network related member variables to default disconnected state.
		/// </summary>
		private void ResetConnectionStateVars()
		{
			_client = null;
			_stream = null;
			_ackTimeoutTimer = null;
			_requestSent = false;
			SetConnectionState( ConnectionState.Disconnected );
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8295"> 
		///		<ExpectedInput>No input parameter required.</ExpectedInput>
		///		<ExpectedOutput>HL7Interface object returned is the same as the one used in constructor.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8296"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Remote server connection information.
		/// </summary>
		public HL7Interface InterfaceConnectionInfo
		{
			get
			{
				return _interfaceConnectInfo;
			}
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8297"> 
		///		<ExpectedInput>Not testable. Requires connected remote host.</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8298"> 
		///		<ExpectedInput>Called with no valid connection.</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates whether connection with HL7 Listener is established. 
		/// </summary>
		public bool IsConnected
		{
			get
			{
				lock( this )
					return _connectionState == ConnectionState.Connected;
			}
		}

		/// <summary>
		/// Sets connection state and notifies registered handlers if state change has occured.
		/// </summary>
		/// <param name="connectionStateToSet">New connection state to set.</param>
		private void SetConnectionState( ConnectionState connectionStateToSet )
		{
			lock( this )
			{
				ConnectionState _previousConnectionState = _connectionState;

				_connectionState = connectionStateToSet;

				bool _isNewStateConnected = IsConnectedState( connectionStateToSet );
			}
		}

		/// <summary>
		/// Determines whether a given connection state is available ('connected') state - 
		/// state in which connection may be used by outside callers.
		/// </summary>
		/// <param name="state"></param>
		/// <returns></returns>
		private bool IsConnectedState( ConnectionState state )
		{
			return state == ConnectionState.Connected;
		}
	}
}
